'use strict';

/* --------------------------------------------------------------
 events.js 2019-02-26
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2018 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Main Table Events
 *
 * Handles the events of the main orders table.
 */
gx.controllers.module('events', [jse.source + '/vendor/momentjs/moment.min.js', 'loading_spinner', 'modal', 'xhr'], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------

    /**
     * Module Selector
     *
     * @type {jQuery}
     */

    var $this = $(this);

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {};

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * On Bulk Selection Change
     *
     * @param {jQuery.Event} event jQuery event object.
     * @param {Boolean} propagate Whether to affect the body elements. We do not need this on "draw.dt" event.
     */
    function _onBulkSelectionChange(event) {
        var propagate = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : true;

        if (propagate === false) {
            return; // Do not propagate on draw event because the body checkboxes are unchecked by default.
        }

        $this.find('tbody input:checkbox').single_checkbox('checked', $(this).prop('checked')).trigger('change');
    }

    /**
     * On Table Row Click
     *
     * When a row is clicked then the row-checkbox must be toggled.
     *
     * @param {jQuery.Event} event
     */
    function _onTableRowClick(event) {
        if (!$(event.target).is('td')) {
            return;
        }

        $(this).find('input:checkbox').prop('checked', !$(this).find('input:checkbox').prop('checked')).trigger('change');
    }

    /**
     * On Table Row Checkbox Change
     *
     * Adjust the bulk actions state whenever there are changes in the table checkboxes.
     */
    function _onTableRowCheckboxChange() {
        if ($this.find('input:checkbox:checked').length > 0) {
            $this.parents('.orders').find('.bulk-action > button').removeClass('disabled');
        } else {
            $this.parents('.orders').find('.bulk-action > button').addClass('disabled');
        }
    }

    /**
     * On Cancel Order Click
     *
     * @param {jQuery.Event} event
     */
    function _onCancelOrderClick(event) {
        event.preventDefault();

        var selectedOrders = _getSelectedOrders($(this));

        // Show the order delete modal.
        $('.cancel.modal .selected-orders').text(selectedOrders.join(', '));
        $('.cancel.modal').modal('show');
    }

    /**
     * On Send Order Click.
     *
     * Sends the email order confirmations.
     *
     * @param {jQuery.Event} event jQuery event object.
     */
    function _onBulkEmailOrderClick(event) {
        var $modal = $('.bulk-email-order.modal');
        var $mailList = $modal.find('.email-list');

        var generateMailRowMarkup = function generateMailRowMarkup(data) {
            var $row = $('<div/>', { class: 'form-group email-list-item' });
            var $idColumn = $('<div/>', { class: 'col-sm-3' });
            var $emailColumn = $('<div/>', { class: 'col-sm-9' });

            var $idLabel = $('<label/>', {
                class: 'control-label id-label force-text-color-black force-text-normal-weight',
                text: data.id
            });

            var $emailInput = $('<input/>', {
                class: 'form-control email-input',
                type: 'text',
                value: data.customerEmail
            });

            $idLabel.appendTo($idColumn);
            $emailInput.appendTo($emailColumn);

            $row.append([$idColumn, $emailColumn]);
            $row.data('order', data);

            return $row;
        };

        var selectedOrders = [];

        event.preventDefault();

        $this.find('tbody input:checkbox:checked').each(function () {
            var rowData = $(this).parents('tr').data();
            selectedOrders.push(rowData);
        });

        if (selectedOrders.length) {
            $mailList.empty();
            selectedOrders.forEach(function (order) {
                return $mailList.append(generateMailRowMarkup(order));
            });
            $modal.modal('show');
        }
    }

    /**
     * On Send Invoice Click.
     *
     * Sends the email invoice.
     *
     * @param {jQuery.Event} event Fired event.
     */
    function _onBulkEmailInvoiceClick(event) {
        var $modal = $('.bulk-email-invoice.modal');
        var $mailList = $modal.find('.email-list');

        var generateMailRowMarkup = function generateMailRowMarkup(data) {
            var $row = $('<div/>', { class: 'form-group email-list-item' });
            var $idColumn = $('<div/>', { class: 'col-sm-3' });
            var $invoiceColumn = $('<div/>', { class: 'col-sm-3' });
            var $emailColumn = $('<div/>', { class: 'col-sm-6' });

            var $latestInvoiceIdInput = $('<input/>', {
                class: 'form-control latest-invoice-id',
                type: 'hidden',
                value: data.latestInvoiceId
            });

            var $idLabel = $('<label/>', {
                class: 'control-label id-label force-text-color-black force-text-normal-weight',
                text: data.id
            });

            var $invoiceLink = $('<label/>', {
                class: 'control-label id-label force-text-color-black force-text-normal-weight',
                html: data.latestInvoiceNumber ? '<a href="request_port.php?module=OrderAdmin&action=showPdf&type=invoice' + ('&invoice_number=' + data.latestInvoiceNumber + '&order_id=' + data.id + '" target="_blank">') + (data.latestInvoiceNumber + '</a>') : '-'
            });

            var $emailInput = $('<input/>', {
                class: 'form-control email-input',
                type: 'text',
                value: data.customerEmail
            });

            $idLabel.appendTo($idColumn);
            $invoiceLink.appendTo($invoiceColumn);
            $emailInput.appendTo($emailColumn);

            $row.append([$idColumn, $invoiceColumn, $emailColumn, $latestInvoiceIdInput]);
            $row.data('order', data);

            return $row;
        };

        var selectedInvoice = [];

        event.preventDefault();

        $this.find('tbody input:checkbox:checked').each(function () {
            var rowData = $(this).parents('tr').data();
            selectedInvoice.push(rowData);
        });

        if (selectedInvoice.length) {
            $mailList.empty();
            selectedInvoice.forEach(function (order) {
                return $mailList.append(generateMailRowMarkup(order));
            });
            $modal.modal('show');
        }
    }

    /**
     * Collects the IDs of the selected orders and returns them as an array.
     *
     * @param {jQuery} $target The triggering target
     *
     * @return {Number[]} array of order IDs
     */
    function _getSelectedOrders($target) {
        var selectedOrders = [];

        if ($target.parents('.bulk-action').length > 0) {
            // Fetch the selected order IDs.
            $this.find('tbody input:checkbox:checked').each(function () {
                selectedOrders.push($(this).parents('tr').data('id'));
            });
        } else {
            var rowId = $target.parents('tr').data('id');

            if (!rowId) {
                return; // No order ID was found.
            }

            selectedOrders.push(rowId);
        }

        return selectedOrders;
    }

    /**
     * On Email Invoice Click
     *
     * Display the email-invoice modal.
     */
    function _onEmailInvoiceClick() {
        var $modal = $('.email-invoice.modal');
        var rowData = $(this).parents('tr').data();
        var url = jse.core.config.get('appUrl') + '/admin/admin.php';
        var data = {
            id: rowData.id,
            do: 'OrdersModalsAjax/GetEmailInvoiceSubject',
            pageToken: jse.core.config.get('pageToken')
        };
        var invoiceNumbersHtml = '';

        $modal.find('.customer-info').text('"' + rowData.customerName + '"');
        $modal.find('.email-address').val(rowData.customerEmail);

        $modal.data('orderId', rowData.id).modal('show');

        $.ajax({ url: url, data: data, dataType: 'json' }).done(function (response) {
            $modal.attr('data-gx-widget', 'single_checkbox');

            $modal.find('.subject').val(response.subject);
            if (response.invoiceIdExists) {
                $modal.find('.invoice-num-info').addClass('hidden');
                $modal.find('.no-invoice').removeClass('hidden');
            } else {
                $modal.find('.invoice-num-info').removeClass('hidden');
                $modal.find('.no-invoice').addClass('hidden');
            }

            if (Object.keys(response.invoiceNumbers).length <= 1) {
                $modal.find('.invoice-numbers').addClass('hidden');
            } else {
                $modal.find('.invoice-numbers').removeClass('hidden');
            }

            for (var invoiceId in response.invoiceNumbers) {
                invoiceNumbersHtml += '<p><input type="checkbox" name="invoice_ids[]" value="' + invoiceId + '" checked="checked" class="invoice-numbers-checkbox" /> ' + response.invoiceNumbers[invoiceId] + '</p>';
            }

            $modal.find('.invoice-numbers-checkboxes').html(invoiceNumbersHtml);

            $modal.find('.invoice-numbers-checkbox').on('change', _onChangeEmailInvoiceCheckbox);

            gx.widgets.init($modal);
        });
    }

    /**
     * On Email Invoice Checkbox Change
     *
     * Disable send button if all invoice number checkboxes are unchecked. Otherwise enable the send button again.
     */
    function _onChangeEmailInvoiceCheckbox() {
        var $modal = $('.email-invoice.modal');

        if ($modal.find('.invoice-numbers-checkbox').length > 0) {
            if ($modal.find('.invoice-numbers-checkbox:checked').length > 0) {
                $modal.find('.send').prop('disabled', false);
            } else {
                $modal.find('.send').prop('disabled', true);
            }
        } else {
            $modal.find('.send').prop('disabled', false);
        }
    }

    /**
     * On Email Order Click
     *
     * Display the email-order modal.
     *
     * @param {jQuery.Event} event
     */
    function _onEmailOrderClick(event) {
        var $modal = $('.email-order.modal');
        var rowData = $(this).parents('tr').data();
        var dateFormat = jse.core.config.get('languageCode') === 'de' ? 'DD.MM.YY' : 'MM/DD/YY';

        $modal.find('.customer-info').text('"' + rowData.customerName + '"');
        $modal.find('.subject').val(jse.core.lang.translate('ORDER_SUBJECT', 'gm_order_menu') + rowData.id + jse.core.lang.translate('ORDER_SUBJECT_FROM', 'gm_order_menu') + moment(rowData.purchaseDate.date).format(dateFormat));
        $modal.find('.email-address').val(rowData.customerEmail);

        $modal.data('orderId', rowData.id).modal('show');
    }

    /**
     * On Change Order Status Click
     *
     * Display the change order status modal.
     *
     * @param {jQuery.Event} event
     */
    function _onChangeOrderStatusClick(event) {
        if ($(event.target).hasClass('order-status')) {
            event.stopPropagation();
        }

        var $modal = $('.status.modal');
        var rowData = $(this).parents('tr').data();
        var selectedOrders = _getSelectedOrders($(this));

        $modal.find('#status-dropdown').val(rowData ? rowData.statusId : '');

        $modal.find('#comment').val('');
        $modal.find('#notify-customer, #send-parcel-tracking-code, #send-comment').attr('checked', false).parents('.single-checkbox').removeClass('checked');

        // Show the order change status modal.
        $modal.find('.selected-orders').text(selectedOrders.join(', '));
        $modal.modal('show');
    }

    /**
     * On Add Tracking Number Click
     *
     * @param {jQuery.Event} event
     */
    function _onAddTrackingNumberClick(event) {
        var $modal = $('.add-tracking-number.modal');
        var rowData = $(event.target).parents('tr').data();

        $modal.data('orderId', rowData.id);
        $modal.modal('show');
    }

    /**
     * Opens the gm_pdf_order.php in a new tab with invoices as type $_GET argument.
     *
     * The order ids are passed as a serialized array to the oID $_GET argument.
     */
    function _onBulkDownloadInvoiceClick() {
        var orderIds = [];
        var maxAmountInvoicesBulkPdf = data.maxAmountInvoicesBulkPdf;

        $this.find('tbody input:checkbox:checked').each(function () {
            orderIds.push($(this).parents('tr').data('id'));
        });

        if (orderIds.length > maxAmountInvoicesBulkPdf) {
            var $modal = $('.bulk-error.modal');
            $modal.modal('show');

            var $invoiceMessageContainer = $modal.find('.invoices-message');
            $invoiceMessageContainer.removeClass('hidden');
            $modal.on('hide.bs.modal', function () {
                return $invoiceMessageContainer.addClass('hidden');
            });

            return;
        }

        _createBulkPdf(orderIds, 'invoice');
    }

    /**
     * Opens the gm_pdf_order.php in a new tab with packing slip as type $_GET argument.
     *
     * The order ids are passed as a serialized array to the oID $_GET argument.
     */
    function _onBulkDownloadPackingSlipClick() {
        var orderIds = [];
        var maxAmountPackingSlipsBulkPdf = data.maxAmountPackingSlipsBulkPdf;
        var $modal = void 0;
        var $packingSlipsMessageContainer = void 0;

        $this.find('tbody input:checkbox:checked').each(function () {
            orderIds.push($(this).parents('tr').data('id'));
        });

        if (orderIds.length > maxAmountPackingSlipsBulkPdf) {
            $modal = $('.bulk-error.modal');
            $modal.modal('show');
            $packingSlipsMessageContainer = $modal.find('.packing-slips-message');

            $packingSlipsMessageContainer.removeClass('hidden');

            $modal.on('hide.bs.modal', function () {
                $packingSlipsMessageContainer.addClass('hidden');
            });

            return;
        }

        _createBulkPdf(orderIds, 'packingslip');
    }

    /**
     * Creates a bulk pdf with invoices or packing slips information.
     *
     * This method will check if all the selected orders have a document and open a concatenated PDF file. If there
     * are orders that do not have any document then a modal will be shown, prompting the user to create the missing
     * documents or continue without them.
     *
     * @param {Number[]} orderIds Provide the selected order IDs.
     * @param {String} type Provide the bulk PDF type "invoice" or "packingslip".
     */
    function _createBulkPdf(orderIds, type) {
        if (type !== 'invoice' && type !== 'packingslip') {
            throw new Error('Invalid type provided: ' + type);
        }

        var url = jse.core.config.get('appUrl') + '/admin/admin.php';
        var data = {
            do: 'OrdersOverviewAjax/GetOrdersWithoutDocuments',
            pageToken: jse.core.config.get('pageToken'),
            type: type,
            orderIds: orderIds
        };

        $.getJSON(url, data).done(function (orderIdsWithoutDocument) {
            if (orderIdsWithoutDocument.exception) {
                var title = jse.core.lang.translate('error', 'messages');
                var message = jse.core.lang.translate('GET_ORDERS_WITHOUT_DOCUMENT_ERROR', 'admin_orders');
                jse.libs.modal.showMessage(title, message);
                return;
            }

            if (!orderIdsWithoutDocument.length) {
                _openBulkPdfUrl(orderIds, type); // All the selected order have documents.
                return;
            }

            // Some orders do not have documents, display the confirmation modal.
            var $modal = $('.modal.create-missing-documents');
            $modal.find('.order-ids-list').text(orderIdsWithoutDocument.join(', '));
            $modal.data({
                orderIds: orderIds,
                orderIdsWithoutDocument: orderIdsWithoutDocument,
                type: type
            }).modal('show');
        }).fail(function (jqxhr, textStatus, errorThrown) {
            var title = jse.core.lang.translate('error', 'messages');
            var message = jse.core.lang.translate('GET_ORDERS_WITHOUT_DOCUMENT_ERROR', 'admin_orders');
            jse.libs.modal.showMessage(title, message);
        });
    }

    /**
     * Create Missing Documents Proceed Handler
     *
     * This handler will be executed whenever the user proceeds through the "create-missing-documents" modal. It will
     * be resolved even if the user does not select the checkbox "create-missing-documents".
     *
     * @param {jQuery.Event} event jQuery event object.
     * @param {Number[]} orderIds The selected orders to be included in the PDF.
     * @param {String} type Whether 'invoice' or 'packingslip'.
     * @param {Object} downloadPdfWindow Provide a window handle for bypassing browser's popup blocking.
     */
    function _onCreateMissingDocumentsProceed(event, orderIds, type, downloadPdfWindow) {
        _openBulkPdfUrl(orderIds, type, downloadPdfWindow);
    }

    /**
     * On Single Checkbox Ready
     *
     * This callback will use the event.data.orderIds to set the checked checkboxes after a table re-render.
     *
     * @param {jQuery.Event} event
     */
    function _onSingleCheckboxReady(event) {
        event.data.orderIds.forEach(function (id) {
            $this.find('tr#' + id + ' input:checkbox').single_checkbox('checked', true).trigger('change');
        });

        // Bulk action button should't be disabled after a datatable reload.
        if ($('tr input:checkbox:checked').length) {
            $('.bulk-action').find('button').removeClass('disabled');
        }
    }

    /**
     * Opens the URL which provides the bulk PDF for download.
     *
     * @param {Number[]} orderIds The orders to be used for the concatenated document.
     * @param {String} type Whether 'invoice' or 'packingslip'.
     */
    function _openBulkPdfUrl(orderIds, type) {
        var parameters = {
            do: 'OrdersModalsAjax/BulkPdf' + (type === 'invoice' ? 'Invoices' : 'PackingSlips'),
            pageToken: jse.core.config.get('pageToken'),
            o: orderIds
        };

        var url = jse.core.config.get('appUrl') + '/admin/admin.php?' + $.param(parameters);

        window.open(url, '_parent');

        // Keep checkboxes checked after a datatable reload.
        $this.DataTable().ajax.reload(function () {
            $this.off('single_checkbox:ready', _onSingleCheckboxReady).on('single_checkbox:ready', { orderIds: orderIds }, _onSingleCheckboxReady);
        });
        $this.orders_overview_filter('reload');
    }

    /**
     * On Packing Slip Click
     */
    function _onShowPackingSlipClick() {
        // Message modal data.
        var title = jse.core.lang.translate('TITLE_SHOW_PACKINGSLIP', 'orders');
        var message = jse.core.lang.translate('NO_PACKINGSLIP_AVAILABLE', 'orders');

        // Request data.
        var rowData = $(this).parents('tr').data();
        var url = jse.core.config.get('appUrl') + '/admin/admin.php';

        // Request parameters.
        var data = {
            id: rowData.id,
            do: 'OrdersOverviewAjax/GetLatestPackingSlip',
            pageToken: jse.core.config.get('pageToken')
        };

        // Directly open a new tab (popup blocker workaround)
        var newTab = window.open('about:blank');

        $.ajax({ dataType: 'json', url: url, data: data }).done(function (response) {
            if (response.length) {
                // Get the file name from the response.
                var filename = response[0].file;

                // Packing slip link parameters.
                var parameters = {
                    module: 'OrderAdmin',
                    action: 'showPdf',
                    type: 'packingslip',
                    file: filename
                };

                // Open package slip.
                newTab.location = jse.core.config.get('appUrl') + '/admin/request_port.php?' + $.param(parameters);
            } else {
                // No packing slip found
                newTab.close();
                jse.libs.modal.showMessage(title, message);
            }
        });
    }

    /**
     * On Invoice Create Click
     */
    function _onCreateInvoiceClick() {
        var link = $(this).attr('href');
        var $loadingSpinner = jse.libs.loading_spinner.show($this);
        var pageToken = jse.core.config.get('pageToken');
        var orderId = $(this).parents('tr').data().id;
        var url = jse.core.config.get('appUrl') + ('/admin/admin.php?do=OrdersModalsAjax/GetInvoiceCount&pageToken=' + pageToken + '&orderId=' + orderId);

        // Directly open a new tab (popup blocker workaround)
        var newTab = window.open('about:blank');

        function createInvoice() {
            newTab.location = link;
            $this.DataTable().ajax.reload(null, false);
        }

        function addInvoice() {
            window.open(link, '_blank');
            $this.DataTable().ajax.reload(null, false);
        }

        function onRequestSuccess(response) {
            var modalTitle = jse.core.lang.translate('TITLE_CREATE_INVOICE', 'orders');
            var modalMessage = jse.core.lang.translate('TEXT_CREATE_INVOICE_CONFIRMATION', 'orders');
            var modalButtons = [{
                title: jse.core.lang.translate('yes', 'buttons'),
                callback: function callback(event) {
                    closeModal(event);
                    addInvoice();
                }
            }, {
                title: jse.core.lang.translate('no', 'buttons'),
                callback: closeModal
            }];

            function closeModal(event) {
                $(event.target).parents('.modal').modal('hide');
            }

            jse.libs.loading_spinner.hide($loadingSpinner);

            if (!response.count) {
                createInvoice();
            } else {
                newTab.close();
                jse.libs.modal.showMessage(modalTitle, modalMessage, modalButtons);
            }
        }

        function onRequestFailure() {
            jse.libs.loading_spinner.hide($loadingSpinner);
            createInvoice();
        }

        jse.libs.xhr.get({ url: url }).done(onRequestSuccess).fail(onRequestFailure);
    }

    /**
     * On Invoice Link Click
     *
     * The script that generates the PDFs is changing the status of an order to "invoice-created". Thus the
     * table data need to be redrawn and the filter options to be updated.
     */
    function _onShowInvoiceClick() {
        // Message modal data.
        var title = jse.core.lang.translate('TITLE_SHOW_INVOICE', 'orders');
        var message = jse.core.lang.translate('NO_INVOICE_AVAILABLE', 'orders');

        // Request data.
        var rowData = $(this).parents('tr').data();
        var url = jse.core.config.get('appUrl') + '/admin/admin.php';

        // Request parameters.
        var data = {
            id: rowData.id,
            do: 'OrdersOverviewAjax/GetInvoices',
            pageToken: jse.core.config.get('pageToken')
        };

        // Directly open a new tab (popup blocker workaround)
        var newTab = window.open('about:blank');

        $.ajax({ dataType: 'json', url: url, data: data }).done(function (response) {
            if (response.length) {
                // Get the file name from object with the highest ID within response array.
                var _response$ = response[0],
                    invoiceNumber = _response$.invoiceNumber,
                    orderId = _response$.orderId;

                // Invoice link parameters.

                var parameters = {
                    module: 'OrderAdmin',
                    action: 'showPdf',
                    type: 'invoice',
                    invoice_number: invoiceNumber,
                    order_id: orderId
                };

                // Open invoice
                newTab.location = jse.core.config.get('appUrl') + '/admin/request_port.php?' + $.param(parameters);
            } else {
                // No invoice found
                newTab.close();
                jse.libs.modal.showMessage(title, message);
            }
        });
    }

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        // Bind table row actions.
        $this.on('click', 'tbody tr', _onTableRowClick).on('change', '.bulk-selection', _onBulkSelectionChange).on('change', 'input:checkbox', _onTableRowCheckboxChange).on('click', '.show-invoice', _onShowInvoiceClick).on('click', '.show-packing-slip', _onShowPackingSlipClick).on('click', '.create-invoice', _onCreateInvoiceClick).on('click', '.email-invoice', _onEmailInvoiceClick).on('click', '.email-order', _onEmailOrderClick).on('click', '.order-status.label', _onChangeOrderStatusClick).on('click', '.add-tracking-number', _onAddTrackingNumberClick);

        // Bind table row and bulk actions.
        $this.parents('.orders').on('click', '.btn-group .change-status', _onChangeOrderStatusClick).on('click', '.btn-group .cancel', _onCancelOrderClick).on('click', '.btn-group .bulk-email-order', _onBulkEmailOrderClick).on('click', '.btn-group .bulk-email-invoice', _onBulkEmailInvoiceClick).on('click', '.btn-group .bulk-download-invoice', _onBulkDownloadInvoiceClick).on('click', '.btn-group .bulk-download-packing-slip', _onBulkDownloadPackingSlipClick);

        // Bind custom events.
        $(document).on('create_missing_documents:proceed', '.modal.create-missing-documents', _onCreateMissingDocumentsProceed);

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
